extern crate ctrlc;
use atomic_enum::atomic_enum;
use chacha20poly1305::Error;
use chacha20poly1305::{
    aead::{Aead, AeadCore, KeyInit, OsRng},
    XChaCha20Poly1305, XNonce,
};
use std::os::unix::io::{AsRawFd, RawFd};
use termios::*;
use tokio::fs::OpenOptions;
use tokio::io::AsyncReadExt;
use tokio::io::AsyncWrite;
use tokio::io::AsyncWriteExt;
use tokio::time::timeout;
extern crate simplelog;
use simplelog::*;
mod asyncfile;
use crate::asyncfile::AsyncFile;
use futures::future::join_all;
use rppal::gpio::Gpio;
use rppal::gpio::Trigger;
use std::env;
use std::io;
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::{Arc, RwLock};
use std::thread;
use std::time::{Duration, Instant};
use tokio::task;

use futures::stream::StreamExt;
use std::str;

use tokio_serial::SerialPortBuilderExt;
use tokio_serial::SerialStream;

// Just a generic Result type to ease error handling for us. Errors in multithreaded
// async contexts needs some extra restrictions
type Resulta<T> = std::result::Result<T, Box<dyn std::error::Error + Send + Sync>>;

fn logging_init(debug: bool) {
    let conf = ConfigBuilder::new()
        .set_time_format("%F, %H:%M:%S%.3f".to_string())
        .set_write_log_enable_colors(true)
        .build();

    let mut loggers = vec![];

    let console_logger: Box<dyn SharedLogger> = TermLogger::new(
        if debug {
            LevelFilter::Debug
        } else {
            LevelFilter::Info
        },
        conf.clone(),
        TerminalMode::Mixed,
        ColorChoice::Auto,
    );
    loggers.push(console_logger);

    CombinedLogger::init(loggers).expect("Cannot initialize logging subsystem");
}

pub struct Remeha {
    pub display_name: String,
    pub poll_ok: u64,
    pub poll_errors: u64,
    pub run_mode: RunMode,
}
pub const REMEHA_POLL_INTERVAL_SECS: f32 = 5.0; //secs between polling
pub const REMEHA_STATS_DUMP_INTERVAL_SECS: f32 = 3600.0; //secs between showing stats

impl Remeha {
    pub async fn worker(&mut self, state: Arc<AtomicState>) -> Resulta<()> {
        info!("{} Starting task", self.display_name);
        let mut poll_interval = Instant::now();
        let mut stats_interval = Instant::now();
        let mut terminated = false;

        loop {
            if terminated || state.load(Ordering::SeqCst) == State::Terminating {
                break;
            }
            tokio::time::sleep(Duration::from_millis(30)).await;
        }

        info!("{} task stopped", self.display_name);
        Ok(())
    }
}

const TEST_PIN: u8 = 4; //BCM pin #

#[atomic_enum]
#[derive(PartialEq)]
enum State {
    Idle,
    ButtonPressed,
    Processing,
    Terminating,
}

#[derive(PartialEq)]
enum RunMode {
    Help,
    Client,
    Server,
}

#[tokio::main]
async fn main() -> Result<(), Error> {
    let mut futures = vec![];

    logging_init(false);
    info!("<b><blue>lora</> started");

    let argument = env::args().nth(1);
    let run_mode = if let Some(arg) = argument {
        match &arg[..] {
            "client" => RunMode::Client,
            "server" => RunMode::Server,
            _ => RunMode::Help,
        }
    } else {
        RunMode::Help
    };

    if run_mode == RunMode::Help {
        println! {"No arguments given, possible choices: [ client | server ]\n"};
        return Ok(());
    };

    //main program state
    let state = Arc::new(AtomicState::new(State::Idle));

    //Ctrl-C / SIGTERM support
    let s = state.clone();
    ctrlc::set_handler(move || {
        s.store(State::Terminating, Ordering::SeqCst);
    })
    .expect("Error setting Ctrl-C handler");

    let g;
    let mut pin;
    if run_mode == RunMode::Client {
        let s = state.clone();
        // Retrieve the GPIO pin and configure it as an output.
        g = Gpio::new().expect("GPIO setup problem");
        pin = g
            .get(TEST_PIN)
            .expect("TEST PIN get problem")
            .into_input_pullup();
        let _ = pin.set_async_interrupt(
            Trigger::FallingEdge,
            Some(Duration::from_millis(5)),
            move |_| {
                if s.load(Ordering::SeqCst) == State::Idle {
                    info!("button press!");
                    s.store(State::ButtonPressed, Ordering::SeqCst);
                } else {
                    info!("button press ignored!");
                }
            },
        );
    }

    let s = state.clone();
    let mut remeha = self::Remeha {
        display_name: "<i><bright-black>lora:</>".to_string(),
        poll_ok: 0,
        poll_errors: 0,
        run_mode,
    };

    let remeha_future = task::spawn(async move { remeha.worker(s).await });
    futures.push(remeha_future);

    //wait for tokio async tasks
    let _ = join_all(futures).await;

    info!("🚩 program terminated");
    Ok(())
}
